//--------------------------------------------------------------//
// Shapes
//
// Copyright (c) LWKS Software Ltd.  All Rights Reserved
//--------------------------------------------------------------//

int _LwksEffectInfo
<
   string EffectGroup = "GenericPixelShader";
   string Description = "Simple 2D Shape";
   string Category    = "Mattes";
   bool CanSize       = true;
> = 0;


//--------------------------------------------------------------//
// Params
//--------------------------------------------------------------//
int SetTechnique
<
   string Description = "Shape";
   string Enum = "Rectangle,Ellipse";
> = 0;

float CentreX
<
   string Description = "Position";
   string Flags = "SpecifiesPointX";
   float MinVal = -1.00;
   float MaxVal = 2.00;
> = 0.5;

float CentreY
<
   string Description = "Position";
   string Flags = "SpecifiesPointY";
   float MinVal = -1.00;
   float MaxVal = 2.00;
> = 0.5;

float Width
<
   string Description = "Width";
   float MinVal = 0.00;
   float MaxVal = 1.00;
> = 0.5;

float Height
<
   string Description = "Height";
   float MinVal = 0.00;
   float MaxVal = 1.00;
> = 0.5;

float Softness
<
   string Description = "Softness";
   float MinVal = 0.00;
   float MaxVal = 1.00;
> = 0.1;

float4 FGColour
<
   string Description = "Foreground";
   bool SupportsAlpha = true;
> = { 1.0, 1.0, 1.0, 1.0 };

float4 BGColour
<
   string Description = "Background";
   bool SupportsAlpha = true;
> = { 0.0, 0.0, 0.0, 0.0 };

float _OutputAspectRatio = 1.0;

//--------------------------------------------------------------//
// Code
//--------------------------------------------------------------//
float4 rectangle_main( float2 xyNorm : TEXCOORD0 ) : COLOR
{
   float sx  = ( Softness * Width ) / _OutputAspectRatio;
   float sy  = ( Softness * Width );
   float halfSx = sx / 2.0;
   float halfSy = sy / 2.0;

   // Calc exact rectangle bounds
   float l = CentreX - ( Width / _OutputAspectRatio / 2.0 );
   float r = l + Width / _OutputAspectRatio;
   float t = ( 1.0 - CentreY ) - ( Height / 2.0 );
   float b = t + Height;

   // Now calc inner bounds (ie. taking softness into account)
   float l1 = l + halfSx;
   float r1 = r - halfSx;
   float t1 = t + halfSy;
   float b1 = b - halfSy;

   // And outer bounds
   float l2 = l - halfSx;
   float r2 = r + halfSx;
   float t2 = t - halfSy;
   float b2 = b + halfSy;

   float amt = 1.0;

   if ( xyNorm.x >=l1 && xyNorm.x <= r1 && xyNorm.y >= t1 && xyNorm.y <= b1 )
   {
   }
   else if ( xyNorm.x < l2 || xyNorm.x > r2 || xyNorm.y < t2 || xyNorm.y > b2 )
   {
      amt = 0.0;
   }
   else
   {
      if ( xyNorm.x < l1 )
      {
         amt = ( xyNorm.x - l2 ) / sx;
      }
      else if ( xyNorm.x > r1 )
      {
         amt = ( r2 - xyNorm.x ) / sx;
      }

      if ( xyNorm.y < t1 )
      {
         amt = min( ( xyNorm.y - t2 ) / sy, amt );
      }
      else if ( xyNorm.y > b1 )
      {
         amt = min( ( b2 - xyNorm.y ) / sy, amt );
      }
   }

   return lerp( BGColour, FGColour, amt );
}
//--------------------------------------------------------------//
float4 ellipse_main( float2 xyNorm : TEXCOORD0 ) : COLOR
{
   float4 ret;

   float radius = 0.5;
   float centreY = 1.0 - CentreY;
   float halfSoftness = Softness / 2.0;

   float2 aspectAdjustedpos = float2( ( ( xyNorm.x - CentreX ) / Width ) + CentreX,
                                      ( ( xyNorm.y - centreY ) / Height / _OutputAspectRatio ) + centreY );

   float dist = distance( aspectAdjustedpos, float2( CentreX, centreY ) );

   if ( dist < ( radius - halfSoftness ) )
   {
      ret = FGColour;
   }
   else if ( dist > ( radius + halfSoftness ) )
   {
      ret = BGColour;
   }
   else
   {
      ret = lerp( FGColour, BGColour, ( dist - ( radius - halfSoftness ) ) / Softness );
   }

   return ret;
}

//--------------------------------------------------------------//
// Technique Section for Image Processing Effects
//--------------------------------------------------------------//
technique Rectangle { pass Single_Pass { PixelShader = compile PROFILE rectangle_main(); } }
technique Ellipse   { pass Single_Pass { PixelShader = compile PROFILE ellipse_main(); } }
