﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/sts/STS_EXPORTS.h>
#include <aws/sts/model/Credentials.h>
#include <aws/sts/model/ResponseMetadata.h>

#include <utility>

namespace Aws {
template <typename RESULT_TYPE>
class AmazonWebServiceResult;

namespace Utils {
namespace Xml {
class XmlDocument;
}  // namespace Xml
}  // namespace Utils
namespace STS {
namespace Model {
class AssumeRootResult {
 public:
  AWS_STS_API AssumeRootResult() = default;
  AWS_STS_API AssumeRootResult(const Aws::AmazonWebServiceResult<Aws::Utils::Xml::XmlDocument>& result);
  AWS_STS_API AssumeRootResult& operator=(const Aws::AmazonWebServiceResult<Aws::Utils::Xml::XmlDocument>& result);

  ///@{
  /**
   * <p>The temporary security credentials, which include an access key ID, a secret
   * access key, and a security token.</p>  <p>The size of the security token
   * that STS API operations return is not fixed. We strongly recommend that you make
   * no assumptions about the maximum size.</p>
   */
  inline const Credentials& GetCredentials() const { return m_credentials; }
  template <typename CredentialsT = Credentials>
  void SetCredentials(CredentialsT&& value) {
    m_credentialsHasBeenSet = true;
    m_credentials = std::forward<CredentialsT>(value);
  }
  template <typename CredentialsT = Credentials>
  AssumeRootResult& WithCredentials(CredentialsT&& value) {
    SetCredentials(std::forward<CredentialsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The source identity specified by the principal that is calling the
   * <code>AssumeRoot</code> operation.</p> <p>You can use the
   * <code>aws:SourceIdentity</code> condition key to control access based on the
   * value of source identity. For more information about using source identity, see
   * <a
   * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_temp_control-access_monitor.html">Monitor
   * and control actions taken with assumed roles</a> in the <i>IAM User
   * Guide</i>.</p> <p>The regex used to validate this parameter is a string of
   * characters consisting of upper- and lower-case alphanumeric characters with no
   * spaces. You can also include underscores or any of the following characters:
   * =,.@-</p>
   */
  inline const Aws::String& GetSourceIdentity() const { return m_sourceIdentity; }
  template <typename SourceIdentityT = Aws::String>
  void SetSourceIdentity(SourceIdentityT&& value) {
    m_sourceIdentityHasBeenSet = true;
    m_sourceIdentity = std::forward<SourceIdentityT>(value);
  }
  template <typename SourceIdentityT = Aws::String>
  AssumeRootResult& WithSourceIdentity(SourceIdentityT&& value) {
    SetSourceIdentity(std::forward<SourceIdentityT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const ResponseMetadata& GetResponseMetadata() const { return m_responseMetadata; }
  template <typename ResponseMetadataT = ResponseMetadata>
  void SetResponseMetadata(ResponseMetadataT&& value) {
    m_responseMetadataHasBeenSet = true;
    m_responseMetadata = std::forward<ResponseMetadataT>(value);
  }
  template <typename ResponseMetadataT = ResponseMetadata>
  AssumeRootResult& WithResponseMetadata(ResponseMetadataT&& value) {
    SetResponseMetadata(std::forward<ResponseMetadataT>(value));
    return *this;
  }
  ///@}
 private:
  Credentials m_credentials;

  Aws::String m_sourceIdentity;

  ResponseMetadata m_responseMetadata;
  bool m_credentialsHasBeenSet = false;
  bool m_sourceIdentityHasBeenSet = false;
  bool m_responseMetadataHasBeenSet = false;
};

}  // namespace Model
}  // namespace STS
}  // namespace Aws
