﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/sagemaker/SageMakerRequest.h>
#include <aws/sagemaker/SageMaker_EXPORTS.h>
#include <aws/sagemaker/model/ListLabelingJobsForWorkteamSortByOptions.h>
#include <aws/sagemaker/model/SortOrder.h>

#include <utility>

namespace Aws {
namespace SageMaker {
namespace Model {

/**
 */
class ListLabelingJobsForWorkteamRequest : public SageMakerRequest {
 public:
  AWS_SAGEMAKER_API ListLabelingJobsForWorkteamRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListLabelingJobsForWorkteam"; }

  AWS_SAGEMAKER_API Aws::String SerializePayload() const override;

  AWS_SAGEMAKER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the work team for which you want to see
   * labeling jobs for.</p>
   */
  inline const Aws::String& GetWorkteamArn() const { return m_workteamArn; }
  inline bool WorkteamArnHasBeenSet() const { return m_workteamArnHasBeenSet; }
  template <typename WorkteamArnT = Aws::String>
  void SetWorkteamArn(WorkteamArnT&& value) {
    m_workteamArnHasBeenSet = true;
    m_workteamArn = std::forward<WorkteamArnT>(value);
  }
  template <typename WorkteamArnT = Aws::String>
  ListLabelingJobsForWorkteamRequest& WithWorkteamArn(WorkteamArnT&& value) {
    SetWorkteamArn(std::forward<WorkteamArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of labeling jobs to return in each page of the
   * response.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListLabelingJobsForWorkteamRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If the result of the previous <code>ListLabelingJobsForWorkteam</code>
   * request was truncated, the response includes a <code>NextToken</code>. To
   * retrieve the next set of labeling jobs, use the token in the next request.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListLabelingJobsForWorkteamRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A filter that returns only labeling jobs created after the specified time
   * (timestamp).</p>
   */
  inline const Aws::Utils::DateTime& GetCreationTimeAfter() const { return m_creationTimeAfter; }
  inline bool CreationTimeAfterHasBeenSet() const { return m_creationTimeAfterHasBeenSet; }
  template <typename CreationTimeAfterT = Aws::Utils::DateTime>
  void SetCreationTimeAfter(CreationTimeAfterT&& value) {
    m_creationTimeAfterHasBeenSet = true;
    m_creationTimeAfter = std::forward<CreationTimeAfterT>(value);
  }
  template <typename CreationTimeAfterT = Aws::Utils::DateTime>
  ListLabelingJobsForWorkteamRequest& WithCreationTimeAfter(CreationTimeAfterT&& value) {
    SetCreationTimeAfter(std::forward<CreationTimeAfterT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A filter that returns only labeling jobs created before the specified time
   * (timestamp).</p>
   */
  inline const Aws::Utils::DateTime& GetCreationTimeBefore() const { return m_creationTimeBefore; }
  inline bool CreationTimeBeforeHasBeenSet() const { return m_creationTimeBeforeHasBeenSet; }
  template <typename CreationTimeBeforeT = Aws::Utils::DateTime>
  void SetCreationTimeBefore(CreationTimeBeforeT&& value) {
    m_creationTimeBeforeHasBeenSet = true;
    m_creationTimeBefore = std::forward<CreationTimeBeforeT>(value);
  }
  template <typename CreationTimeBeforeT = Aws::Utils::DateTime>
  ListLabelingJobsForWorkteamRequest& WithCreationTimeBefore(CreationTimeBeforeT&& value) {
    SetCreationTimeBefore(std::forward<CreationTimeBeforeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A filter the limits jobs to only the ones whose job reference code contains
   * the specified string.</p>
   */
  inline const Aws::String& GetJobReferenceCodeContains() const { return m_jobReferenceCodeContains; }
  inline bool JobReferenceCodeContainsHasBeenSet() const { return m_jobReferenceCodeContainsHasBeenSet; }
  template <typename JobReferenceCodeContainsT = Aws::String>
  void SetJobReferenceCodeContains(JobReferenceCodeContainsT&& value) {
    m_jobReferenceCodeContainsHasBeenSet = true;
    m_jobReferenceCodeContains = std::forward<JobReferenceCodeContainsT>(value);
  }
  template <typename JobReferenceCodeContainsT = Aws::String>
  ListLabelingJobsForWorkteamRequest& WithJobReferenceCodeContains(JobReferenceCodeContainsT&& value) {
    SetJobReferenceCodeContains(std::forward<JobReferenceCodeContainsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The field to sort results by. The default is <code>CreationTime</code>.</p>
   */
  inline ListLabelingJobsForWorkteamSortByOptions GetSortBy() const { return m_sortBy; }
  inline bool SortByHasBeenSet() const { return m_sortByHasBeenSet; }
  inline void SetSortBy(ListLabelingJobsForWorkteamSortByOptions value) {
    m_sortByHasBeenSet = true;
    m_sortBy = value;
  }
  inline ListLabelingJobsForWorkteamRequest& WithSortBy(ListLabelingJobsForWorkteamSortByOptions value) {
    SetSortBy(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The sort order for results. The default is <code>Ascending</code>.</p>
   */
  inline SortOrder GetSortOrder() const { return m_sortOrder; }
  inline bool SortOrderHasBeenSet() const { return m_sortOrderHasBeenSet; }
  inline void SetSortOrder(SortOrder value) {
    m_sortOrderHasBeenSet = true;
    m_sortOrder = value;
  }
  inline ListLabelingJobsForWorkteamRequest& WithSortOrder(SortOrder value) {
    SetSortOrder(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_workteamArn;

  int m_maxResults{0};

  Aws::String m_nextToken;

  Aws::Utils::DateTime m_creationTimeAfter{};

  Aws::Utils::DateTime m_creationTimeBefore{};

  Aws::String m_jobReferenceCodeContains;

  ListLabelingJobsForWorkteamSortByOptions m_sortBy{ListLabelingJobsForWorkteamSortByOptions::NOT_SET};

  SortOrder m_sortOrder{SortOrder::NOT_SET};
  bool m_workteamArnHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_creationTimeAfterHasBeenSet = false;
  bool m_creationTimeBeforeHasBeenSet = false;
  bool m_jobReferenceCodeContainsHasBeenSet = false;
  bool m_sortByHasBeenSet = false;
  bool m_sortOrderHasBeenSet = false;
};

}  // namespace Model
}  // namespace SageMaker
}  // namespace Aws
